/*
 * @file drawChar.c
 *
 * @author Matthew Matz
 *
 * @version 0.5
 *
 * @copyright Copyright (C) Parallax, Inc. 2019.  See end of file for
 * terms of use (MIT License).
 *
 * @brief This is a driver that allows the Propeller Multicore Microcontroller to 
 * draw text, shapes, and bitmap files on various screens and displays.
 *
 * @detail This helper-library allows the Propeller Multicore Microcontroller to 
 * draw pixels, lines, circles, recatngles, rounded rectagles, triangles, formatted text 
 * in multiple fonts, bitmap images stored on an SD card on various screens and displays. 
 * At Parallax, we would like to thank Adafruit Industries as parts of this library 
 * were dervied from the Adafruit GFX library for Arduino.  Please submit bug reports, 
 * suggestions, and improvements to this code to editor@parallax.com.
 * 
 * @note If fonts are installed, they occupy EEPROM addresses 40576 to 63359.
 */



#include "simplegfx.h"


const char gfxFontSmall[] = {
  0x00, 0x00, 0x5F, 0x00, 0x00,
  0x00, 0x07, 0x00, 0x07, 0x00,
  0x14, 0x7F, 0x14, 0x7F, 0x14,
  0x24, 0x2A, 0x7F, 0x2A, 0x12,
  0x23, 0x13, 0x08, 0x64, 0x62,
  0x36, 0x49, 0x56, 0x20, 0x50,
  0x00, 0x08, 0x07, 0x03, 0x00,
  0x00, 0x1C, 0x22, 0x41, 0x00,
  0x00, 0x41, 0x22, 0x1C, 0x00,
  0x2A, 0x1C, 0x7F, 0x1C, 0x2A,
  0x08, 0x08, 0x3E, 0x08, 0x08,
  0x00, 0x80, 0x70, 0x30, 0x00,
  0x08, 0x08, 0x08, 0x08, 0x08,
  0x00, 0x00, 0x60, 0x60, 0x00,
  0x20, 0x10, 0x08, 0x04, 0x02,
  0x3E, 0x51, 0x49, 0x45, 0x3E,
  0x00, 0x42, 0x7F, 0x40, 0x00,
  0x72, 0x49, 0x49, 0x49, 0x46,
  0x21, 0x41, 0x49, 0x4D, 0x33,
  0x18, 0x14, 0x12, 0x7F, 0x10,
  0x27, 0x45, 0x45, 0x45, 0x39,
  0x3C, 0x4A, 0x49, 0x49, 0x31,
  0x41, 0x21, 0x11, 0x09, 0x07,
  0x36, 0x49, 0x49, 0x49, 0x36,
  0x46, 0x49, 0x49, 0x29, 0x1E,
  0x00, 0x00, 0x14, 0x00, 0x00,
  0x00, 0x40, 0x34, 0x00, 0x00,
  0x00, 0x08, 0x14, 0x22, 0x41,
  0x14, 0x14, 0x14, 0x14, 0x14,
  0x00, 0x41, 0x22, 0x14, 0x08,
  0x02, 0x01, 0x59, 0x09, 0x06,
  0x3E, 0x41, 0x5D, 0x59, 0x4E,
  0x7C, 0x12, 0x11, 0x12, 0x7C,
  0x7F, 0x49, 0x49, 0x49, 0x36,
  0x3E, 0x41, 0x41, 0x41, 0x22,
  0x7F, 0x41, 0x41, 0x41, 0x3E,
  0x7F, 0x49, 0x49, 0x49, 0x41,
  0x7F, 0x09, 0x09, 0x09, 0x01,
  0x3E, 0x41, 0x41, 0x51, 0x73,
  0x7F, 0x08, 0x08, 0x08, 0x7F,
  0x00, 0x41, 0x7F, 0x41, 0x00,
  0x20, 0x40, 0x41, 0x3F, 0x01,
  0x7F, 0x08, 0x14, 0x22, 0x41,
  0x7F, 0x40, 0x40, 0x40, 0x40,
  0x7F, 0x02, 0x1C, 0x02, 0x7F,
  0x7F, 0x04, 0x08, 0x10, 0x7F,
  0x3E, 0x41, 0x41, 0x41, 0x3E,
  0x7F, 0x09, 0x09, 0x09, 0x06,
  0x3E, 0x41, 0x51, 0x21, 0x5E,
  0x7F, 0x09, 0x19, 0x29, 0x46,
  0x26, 0x49, 0x49, 0x49, 0x32,
  0x03, 0x01, 0x7F, 0x01, 0x03,
  0x3F, 0x40, 0x40, 0x40, 0x3F,
  0x1F, 0x20, 0x40, 0x20, 0x1F,
  0x3F, 0x40, 0x38, 0x40, 0x3F,
  0x63, 0x14, 0x08, 0x14, 0x63,
  0x03, 0x04, 0x78, 0x04, 0x03,
  0x61, 0x59, 0x49, 0x4D, 0x43,
  0x00, 0x7F, 0x41, 0x41, 0x41,
  0x02, 0x04, 0x08, 0x10, 0x20,
  0x00, 0x41, 0x41, 0x41, 0x7F,
  0x04, 0x02, 0x01, 0x02, 0x04,
  0x40, 0x40, 0x40, 0x40, 0x40,
  0x00, 0x03, 0x07, 0x08, 0x00,
  0x20, 0x54, 0x54, 0x78, 0x40,
  0x7F, 0x28, 0x44, 0x44, 0x38,
  0x38, 0x44, 0x44, 0x44, 0x28,
  0x38, 0x44, 0x44, 0x28, 0x7F,
  0x38, 0x54, 0x54, 0x54, 0x18,
  0x00, 0x08, 0x7E, 0x09, 0x02,
  0x18, 0xA4, 0xA4, 0x9C, 0x78,
  0x7F, 0x08, 0x04, 0x04, 0x78,
  0x00, 0x44, 0x7D, 0x40, 0x00,
  0x20, 0x40, 0x40, 0x3D, 0x00,
  0x7F, 0x10, 0x28, 0x44, 0x00,
  0x00, 0x41, 0x7F, 0x40, 0x00,
  0x7C, 0x04, 0x78, 0x04, 0x78,
  0x7C, 0x08, 0x04, 0x04, 0x78,
  0x38, 0x44, 0x44, 0x44, 0x38,
  0xFC, 0x18, 0x24, 0x24, 0x18,
  0x18, 0x24, 0x24, 0x18, 0xFC,
  0x7C, 0x08, 0x04, 0x04, 0x08,
  0x48, 0x54, 0x54, 0x54, 0x24,
  0x04, 0x04, 0x3F, 0x44, 0x24,
  0x3C, 0x40, 0x40, 0x20, 0x7C,
  0x1C, 0x20, 0x40, 0x20, 0x1C,
  0x3C, 0x40, 0x30, 0x40, 0x3C,
  0x44, 0x28, 0x10, 0x28, 0x44,
  0x4C, 0x90, 0x90, 0x90, 0x7C,
  0x44, 0x64, 0x54, 0x4C, 0x44,
  0x00, 0x08, 0x36, 0x41, 0x00,
  0x00, 0x00, 0x77, 0x00, 0x00,
  0x00, 0x41, 0x36, 0x08, 0x00,
  0x02, 0x01, 0x02, 0x04, 0x02,
};


void drawCharSmall(screen_t *dev, unsigned char c) 
{
  c -= 33;

  for (char i = 0; i < 6; i++ ) 
  {
    int li;

    if (i < 5) li = gfxFontSmall[c * 5 + i];
    else       li = 0x0;

    for (char j = 0; j < 8; j++, li >>= 1) 
    {
      if (li & 0x1)                               drawPixel(dev, dev->cursor_x + i, dev->cursor_y + j, dev->text_color);
      //else if (dev->text_color != dev->bg_color)  drawPixel(dev, dev->cursor_x + i, dev->cursor_y + j, dev->bg_color);
    }        
  }
}




void drawCharMedium(screen_t *dev, unsigned char c) 
{
  c -= 33;

  char gfxFontMedium[22];
  i2c_in(dev->eepromBus, 0b1010000, (dev->font[1] + ((unsigned int) c) * 22), 2, gfxFontMedium, 22);

  for(int xy = 0; xy < 22; xy++) if(gfxFontMedium[xy] == 0xEA) gfxFontMedium[xy] = 0x19; 
  
  for (char i = 0; i < 22; i += 2 ) 
  {
    int li;
    
    if (i < 22) li = (gfxFontMedium[i] << 8) | gfxFontMedium[i+1];
    else        li = 0x0;

    for (char j = 0; j < 16; j++, li >>= 1) 
    {
      if (li & 0x1)                               drawPixel(dev, dev->cursor_x + (i >> 1), dev->cursor_y + j, dev->text_color);
      //else if (dev->text_color != dev->bg_color)  drawPixel(dev, dev->cursor_x + (i >> 1), dev->cursor_y + j, dev->bg_color);
    }
  }
} 





void drawCharLarge(screen_t *dev, unsigned char c) 
{
  int offset = 0;
  int idx;
  char li;
  int ctr = 0;
  
  char font_lg_index[95];
  char font_lg_zeroMap[7];
  char gfxFontLarge[51];
  char current_character[51];
  
  memset(current_character, 0, 51);  // set the current character to empty;

  i2c_in(dev->eepromBus, 0b1010000, (dev->font[2] & 0xFFFF), 2, font_lg_index, 95);
  i2c_in(dev->eepromBus, 0b1010000, ((dev->font[3] & 0xFFFF) + 7*(c-33)), 2, font_lg_zeroMap, 7);
  
  for(int i = 1; i < (c-32); i++) 
  {
    if(font_lg_index[i] == 0xEA) font_lg_index[i] = 0x19;
    offset += font_lg_index[i];      // sum the index to find the offset
  }    

  i2c_in(dev->eepromBus, 0b1010000, ((dev->font[4] & 0xFFFF) + offset), 2, gfxFontLarge, 51); 
  
  offset = 0;       
  
  for(int k = 1; k < 8; k++)
  {
    li = font_lg_zeroMap[k - 1];
    if(li == 0xEA) li = 0x19;
    for(int j = 0; j < 8; j++) 
    {
      ctr++;
      char t = li & (1 << (7-j));
      if((t > 0) && ctr < 52)
      {
        current_character[ctr] = gfxFontLarge[offset];
        if(current_character[ctr] == 0xEA) current_character[ctr] = 0x19;
        offset++;
      } else if(ctr < 52) {
        current_character[ctr] = 0x00;
      }
    }                 
  }    

  for (char i = 1; i < 52; i += 3 ) 
  {
    int lj;
    if (i < 51) lj = (current_character[i] << 16) | (current_character[i + 1] << 8) | current_character[i + 2];
    else        lj = 0x0;

    for (char j = 0; j < 23; j++, lj >>= 1) 
    {
      if (lj & 0x1)                               drawPixel(dev, dev->cursor_x + i / 3, dev->cursor_y + j, dev->text_color);
      //else if (dev->text_color != dev->bg_color)  drawPixel(dev, dev->cursor_x + i / 3, dev->cursor_y + j, dev->bg_color);
    }
  }
}



/**
 * TERMS OF USE: MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
